// History management
const HISTORY_KEY = "resume_analysis_history";
const MAX_HISTORY = 20;

function loadHistory() {
  try {
    const history = localStorage.getItem(HISTORY_KEY);
    return history ? JSON.parse(history) : [];
  } catch (e) {
    console.error("Failed to load history:", e);
    return [];
  }
}

function saveToHistory(data) {
  try {
    const history = loadHistory();
    const entry = {
      id: Date.now(),
      timestamp: new Date().toISOString(),
      fileName: selectedFile?.name || "Unknown",
      jobDescriptionPreview: jobDescription.value.substring(0, 100) + "...",
      score: data.initialAnalysis?.score || 0,
      initialAnalysis: data.initialAnalysis,
      tailoredResume: data.tailoredResume,
      finalScore: data.finalScore,
    };

    history.unshift(entry);

    if (history.length > MAX_HISTORY) {
      history.splice(MAX_HISTORY);
    }

    localStorage.setItem(HISTORY_KEY, JSON.stringify(history));
    displayHistoryList();
  } catch (e) {
    console.error("Failed to save to history:", e);
  }
}

function displayHistoryList() {
  const historyList = document.getElementById("history-list");
  const history = loadHistory();

  if (history.length === 0) {
    historyList.innerHTML = '<p class="history-empty">No history yet</p>';
    return;
  }

  historyList.innerHTML = history
    .map((entry) => {
      const date = new Date(entry.timestamp);
      const dateStr = date.toLocaleDateString("en-US", {
        month: "short",
        day: "numeric",
      });
      const timeStr = date.toLocaleTimeString([], {
        hour: "2-digit",
        minute: "2-digit",
      });

      let scoreClass = "score-low";
      if (entry.score >= 75) scoreClass = "score-high";
      else if (entry.score >= 50) scoreClass = "score-medium";

      return `
      <div class="history-item" onclick="loadHistoryEntry(${entry.id})">
        <div class="history-item-header">
          <span class="history-filename">${entry.fileName}</span>
          <span class="history-score ${scoreClass}">${entry.score}</span>
        </div>
        <div class="history-date">${dateStr} at ${timeStr}</div>
      </div>
    `;
    })
    .join("");
}

function loadHistoryEntry(id) {
  const history = loadHistory();
  const entry = history.find((e) => e.id === id);

  if (!entry) {
    showError("History entry not found");
    return;
  }

  // Set the tailored resume content
  tailoredResumeContent = entry.tailoredResume;

  // Display results without re-saving to history
  displayResults(
    {
      initialAnalysis: entry.initialAnalysis,
      tailoredResume: entry.tailoredResume,
      finalScore: entry.finalScore,
    },
    { fromHistory: true },
  );
}

function clearHistory() {
  if (confirm("Are you sure you want to clear all history?")) {
    localStorage.removeItem(HISTORY_KEY);
    displayHistoryList();
  }
}

// DOM Elements
const dropZone = document.getElementById("drop-zone");
const dropZoneContent = document.getElementById("drop-zone-content");
const selectedFileDisplay = document.getElementById("selected-file-display");
const selectedFileName = document.getElementById("selected-file-name");
const changeFileBtn = document.getElementById("change-file-btn");
const jobDescription = document.getElementById("job-description");
const analyzeBtn = document.getElementById("analyze-btn");
const loadingSection = document.getElementById("loading-section");
const resultsSection = document.getElementById("results-section");
const errorToast = document.getElementById("error-toast");
const errorMessage = document.getElementById("error-message");
const previewPanel = document.getElementById("preview-document");

// State
let selectedFile = null;
let tailoredResumeContent = "";

// Live preview - always use the single professional template
async function updateLivePreview() {
  if (!previewPanel || !tailoredResumeContent) return;

  try {
    const result = await window.electronAPI.generatePreviewHTML({
      content: tailoredResumeContent,
    });

    if (result.success && result.html) {
      previewPanel.innerHTML = `
        <iframe 
          style="width: 100%; min-height: 1400px; height: 100%; border: none; background: white; display: block;" 
          srcdoc="${result.html.replace(/"/g, "&quot;")}"
        ></iframe>
      `;
    }
  } catch (error) {
    console.error("Error generating preview:", error);
  }
}

// Simple HTML escape helper
function escapeHtml(text) {
  const div = document.createElement("div");
  div.textContent = text;
  return div.innerHTML;
}

// Right-click context menu for job description
jobDescription.addEventListener("contextmenu", async (e) => {
  e.preventDefault();
  await window.electronAPI.showContextMenu();
});

// Handle paste from context menu
window.electronAPI.onPaste((event, text) => {
  if (document.activeElement === jobDescription) {
    const start = jobDescription.selectionStart;
    const end = jobDescription.selectionEnd;
    const value = jobDescription.value;
    jobDescription.value =
      value.substring(0, start) + text + value.substring(end);
    jobDescription.selectionStart = jobDescription.selectionEnd =
      start + text.length;
    jobDescription.dispatchEvent(new Event("input"));
  }
});

// File Selection
dropZone.addEventListener("click", async (e) => {
  if (e.target === changeFileBtn) return;
  await selectFile();
});

changeFileBtn.addEventListener("click", async (e) => {
  e.stopPropagation();
  await selectFile();
});

async function selectFile() {
  try {
    const result = await window.electronAPI.selectFile();
    if (result) {
      selectedFile = result;
      showSelectedFile(result.fileName);
      validateForm();
    }
  } catch (error) {
    showError("Failed to select file");
  }
}

function showSelectedFile(fileName) {
  dropZoneContent.classList.add("hidden");
  selectedFileDisplay.classList.remove("hidden");
  selectedFileName.textContent = fileName;
  dropZone.style.borderColor = "#10b981";
  dropZone.style.background = "rgba(16, 185, 129, 0.05)";
}

function resetFileSelection() {
  selectedFile = null;
  dropZoneContent.classList.remove("hidden");
  selectedFileDisplay.classList.add("hidden");
  dropZone.style.borderColor = "";
  dropZone.style.background = "";
}

// Form Validation
jobDescription.addEventListener("input", validateForm);

function validateForm() {
  const hasFile = selectedFile !== null;
  const hasJobDescription = jobDescription.value.trim().length > 50;
  analyzeBtn.disabled = !(hasFile && hasJobDescription);
}

// Analyze Resume
analyzeBtn.addEventListener("click", async () => {
  if (!selectedFile || !jobDescription.value.trim()) return;

  // Get selected model
  const modelSelect = document.getElementById("model-select");
  const selectedModel = modelSelect.value;

  // Hide input section and show loading
  document.querySelector(".input-section").style.display = "none";
  resultsSection.classList.add("hidden");
  loadingSection.classList.remove("hidden");

  // Reset loading steps
  resetLoadingSteps();

  try {
    // Step 1: Extracting
    updateLoadingStep(1, "active");
    await delay(500);
    updateLoadingStep(1, "completed");

    // Step 2: Scoring
    updateLoadingStep(2, "active");

    const result = await window.electronAPI.analyzeResume({
      filePath: selectedFile.filePath,
      jobDescription: jobDescription.value.trim(),
      model: selectedModel,
    });

    updateLoadingStep(2, "completed");

    // Step 3: Generating
    updateLoadingStep(3, "active");
    await delay(300);
    updateLoadingStep(3, "completed");

    // Step 4: Final scoring
    updateLoadingStep(4, "active");
    await delay(300);
    updateLoadingStep(4, "completed");

    await delay(500);

    // Display results
    displayResults(result);
  } catch (error) {
    console.error("Analysis error:", error);
    showError(error.message || "Failed to analyze resume");
    document.querySelector(".input-section").style.display = "block";
  } finally {
    loadingSection.classList.add("hidden");
  }
});

function resetLoadingSteps() {
  for (let i = 1; i <= 4; i++) {
    const step = document.getElementById(`step-${i}`);
    step.classList.remove("active", "completed");
  }
}

function updateLoadingStep(stepNumber, status) {
  const step = document.getElementById(`step-${stepNumber}`);
  step.classList.remove("active", "completed");
  step.classList.add(status);
}

function delay(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

// Display Results
function displayResults(data, options = {}) {
  const { fromHistory = false } = options;
  const { initialAnalysis, tailoredResume, finalScore } = data;
  tailoredResumeContent = tailoredResume;

  // Focus on results view
  const inputSection = document.querySelector(".input-section");
  if (inputSection) inputSection.style.display = "none";

  // Before score
  const beforeScore = initialAnalysis.score;
  const beforeKeywordMatch =
    initialAnalysis.keywordAnalysis?.score ||
    initialAnalysis.keywordMatchRate ||
    0;

  animateScore("before-score-value", beforeScore);
  animateScoreCircle("before-score-circle", beforeScore);
  document.getElementById("before-keyword-match").textContent =
    `${beforeKeywordMatch}%`;

  // After score
  const afterScore = finalScore.score;
  const afterKeywordMatch =
    finalScore.keywordAnalysis?.score || finalScore.keywordMatchRate || 0;
  const atsLikelihood = finalScore.atsPassLikelihood || 0;

  animateScore("after-score-value", afterScore);
  animateScoreCircle("after-score-circle", afterScore);
  document.getElementById("after-keyword-match").textContent =
    `${afterKeywordMatch}%`;
  document.getElementById("ats-likelihood").textContent = `${atsLikelihood}%`;

  // Improvement badge
  const improvement = afterScore - beforeScore;
  const improvementBadge = document.getElementById("improvement-badge");
  improvementBadge.textContent =
    improvement >= 0 ? `+${improvement}` : improvement;
  improvementBadge.style.background = improvement >= 0 ? "#10b981" : "#ef4444";

  // Strengths
  const strengthsList = document.getElementById("strengths-list");
  strengthsList.innerHTML = "";
  (initialAnalysis.strengths || []).forEach((strength) => {
    const li = document.createElement("li");
    li.textContent = strength;
    strengthsList.appendChild(li);
  });

  // Improvements
  const improvementsList = document.getElementById("improvements-list");
  improvementsList.innerHTML = "";
  (finalScore.improvements || []).forEach((improvement) => {
    const li = document.createElement("li");
    li.textContent = improvement;
    improvementsList.appendChild(li);
  });

  // Keywords
  const keywordsContainer = document.getElementById("keywords-container");
  keywordsContainer.innerHTML = "";

  const matchedKeywords =
    initialAnalysis.keywordAnalysis?.matchedKeywords || [];
  const addedKeywords =
    finalScore.keywordAnalysis?.matchedKeywords?.filter(
      (k) => !matchedKeywords.includes(k),
    ) || [];

  matchedKeywords.forEach((keyword) => {
    const tag = document.createElement("span");
    tag.className = "keyword-tag matched";
    tag.textContent = keyword;
    keywordsContainer.appendChild(tag);
  });

  addedKeywords.forEach((keyword) => {
    const tag = document.createElement("span");
    tag.className = "keyword-tag added";
    tag.textContent = `+ ${keyword}`;
    keywordsContainer.appendChild(tag);
  });

  // Tailored resume
  const resumeElement = document.getElementById("tailored-resume");
  // Convert **text** to <strong>text</strong> for display
  const formattedResume = tailoredResume
    .replace(/\*\*([^*]+)\*\*/g, "<strong>$1</strong>")
    .replace(/\n/g, "<br>");
  resumeElement.innerHTML = formattedResume;

  // Save to history unless we are reloading an existing entry
  if (
    !fromHistory &&
    data.initialAnalysis &&
    data.tailoredResume &&
    data.finalScore
  ) {
    saveToHistory(data);
  }

  // Show results
  resultsSection.classList.remove("hidden");
  resultsSection.scrollIntoView({ behavior: "smooth", block: "start" });

  updateLivePreview();
}

function animateScore(elementId, targetScore) {
  const element = document.getElementById(elementId);
  let current = 0;
  const duration = 1500;
  const increment = targetScore / (duration / 16);

  const animate = () => {
    current += increment;
    if (current < targetScore) {
      element.textContent = Math.round(current);
      requestAnimationFrame(animate);
    } else {
      element.textContent = targetScore;
    }
  };

  animate();
}

function animateScoreCircle(elementId, score) {
  const circle = document.getElementById(elementId);
  const circumference = 283;
  const offset = circumference - (score / 100) * circumference;

  setTimeout(() => {
    circle.style.strokeDashoffset = offset;
  }, 100);
}

// Copy Resume
document.getElementById("copy-btn").addEventListener("click", async () => {
  try {
    await navigator.clipboard.writeText(tailoredResumeContent);
    showToast("Copied to clipboard!", "success");
  } catch (error) {
    showError("Failed to copy to clipboard");
  }
});

// Save PDF - Now using single professional template
document.getElementById("save-pdf-btn").addEventListener("click", async () => {
  try {
    if (!tailoredResumeContent) {
      showError("No resume content to save");
      return;
    }

    // Show saving indicator
    const btn = document.getElementById("save-pdf-btn");
    const originalText = btn.innerHTML;
    btn.innerHTML =
      '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="animate-spin"><circle cx="12" cy="12" r="10" stroke-opacity="0.25"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke-opacity="0.75"></path></svg> Generating PDF...';
    btn.disabled = true;

    const result = await window.electronAPI.savePDF({
      content: tailoredResumeContent,
    });

    btn.innerHTML = originalText;
    btn.disabled = false;

    if (result.success) {
      showToast("PDF saved successfully!", "success");
    } else if (result.locked) {
      showError(
        result.error ||
          "The PDF file is currently open. Please close it and try again.",
      );
    } else if (result.error) {
      showError(result.error);
    }
  } catch (error) {
    const btn = document.getElementById("save-pdf-btn");
    btn.disabled = false;
    showError("Failed to save PDF: " + (error.message || "Unknown error"));
  }
});

// Save HTML - Download the HTML file that can be opened in browser and printed as PDF
document.getElementById("save-html-btn").addEventListener("click", async () => {
  try {
    if (!tailoredResumeContent) {
      showError("No resume content to save");
      return;
    }

    const btn = document.getElementById("save-html-btn");
    const originalText = btn.innerHTML;
    btn.innerHTML =
      '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="animate-spin"><circle cx="12" cy="12" r="10" stroke-opacity="0.25"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke-opacity="0.75"></path></svg> Generating HTML...';
    btn.disabled = true;

    const result = await window.electronAPI.saveHTML({
      content: tailoredResumeContent,
    });

    btn.innerHTML = originalText;
    btn.disabled = false;

    if (result.success) {
      showToast(
        "HTML saved successfully! Open in browser and use Print > Save as PDF",
        "success",
      );
    } else if (result.error) {
      showError(result.error);
    }
  } catch (error) {
    const btn = document.getElementById("save-html-btn");
    btn.disabled = false;
    showError("Failed to save HTML: " + (error.message || "Unknown error"));
  }
});

// Removed Word, Markdown, and Text save handlers - now only using PDF export

// Start Over
document.getElementById("start-over-btn").addEventListener("click", () => {
  resultsSection.classList.add("hidden");
  document.querySelector(".input-section").style.display = "block";
  resetFileSelection();
  jobDescription.value = "";
  validateForm();
  window.scrollTo({ top: 0, behavior: "smooth" });
});

// Error handling
function showError(message) {
  errorMessage.textContent = message;
  errorToast.classList.remove("hidden");
  setTimeout(() => {
    errorToast.classList.add("hidden");
  }, 5000);
}

function showToast(message, type) {
  // Temporarily reuse error toast for success messages
  const toast = errorToast;
  const msgElement = errorMessage;

  if (type === "success") {
    toast.style.background = "#10b981";
  } else {
    toast.style.background = "#ef4444";
  }

  msgElement.textContent = message;
  toast.classList.remove("hidden");

  setTimeout(() => {
    toast.classList.add("hidden");
    toast.style.background = "";
  }, 3000);
}

// Clear history button
document
  .getElementById("clear-history-btn")
  .addEventListener("click", clearHistory);

// Initialize
validateForm();
displayHistoryList();
