﻿const {
  app,
  BrowserWindow,
  ipcMain,
  dialog,
  Menu,
  clipboard,
} = require("electron");
const path = require("path");
const fs = require("fs").promises;
const OpenAI = require("openai");
const pdfParse = require("pdf-parse");
const mammoth = require("mammoth");
const PDFDocument = require("pdfkit");
const puppeteer = require("puppeteer");
const {
  Document,
  Packer,
  Paragraph,
  TextRun,
  HeadingLevel,
  AlignmentType,
  UnderlineType,
  BorderStyle,
} = require("docx");
require("dotenv").config();

// Initialize OpenAI with GPT-4 for better resume generation
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

let mainWindow;

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    minWidth: 1000,
    minHeight: 700,
    webPreferences: {
      nodeIntegration: false,
      contextIsolation: true,
      preload: path.join(__dirname, "preload.js"),
    },
    icon: path.join(__dirname, "assets", "icon.png"),
    title: "PerfectFit Resume Builder",
    backgroundColor: "#f9fafb",
  });

  mainWindow.loadFile("renderer/index.html");

  // Open DevTools in development
  if (process.env.NODE_ENV === "development") {
    mainWindow.webContents.openDevTools();
  }
}

app.whenReady().then(createWindow);

app.on("window-all-closed", () => {
  if (process.platform !== "darwin") {
    app.quit();
  }
});

app.on("activate", () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow();
  }
});

// Context menu for right-click paste
ipcMain.handle("show-context-menu", async () => {
  const template = [
    {
      label: "Cut",
      role: "cut",
    },
    {
      label: "Copy",
      role: "copy",
    },
    {
      label: "Paste",
      click: () => {
        const text = clipboard.readText();
        mainWindow.webContents.send("context-menu-paste", text);
      },
    },
    { type: "separator" },
    {
      label: "Select All",
      role: "selectAll",
    },
  ];
  const menu = Menu.buildFromTemplate(template);
  menu.popup({ window: mainWindow });
});

// Extract text from PDF
async function extractTextFromPDF(filePath) {
  const dataBuffer = await fs.readFile(filePath);
  const data = await pdfParse(dataBuffer);
  return data.text;
}

// Extract text from DOCX
async function extractTextFromDOCX(filePath) {
  const buffer = await fs.readFile(filePath);
  const result = await mammoth.extractRawText({ buffer: buffer });
  return result.value;
}

// Extract text based on file type
async function extractText(filePath) {
  const ext = path.extname(filePath).toLowerCase();
  if (ext === ".pdf") {
    return await extractTextFromPDF(filePath);
  } else if (ext === ".docx") {
    return await extractTextFromDOCX(filePath);
  }
  throw new Error("Unsupported file type");
}

// Extract keywords from job description
function extractKeywords(jobDescription) {
  // Common technical and soft skills patterns
  const skillPatterns = [
    /\b(python|javascript|typescript|java|c\+\+|c#|ruby|go|rust|swift|kotlin|php|scala|r|sql|html|css|react|angular|vue|node\.?js|express|django|flask|spring|\.net|aws|azure|gcp|docker|kubernetes|git|agile|scrum|ci\/cd|machine learning|deep learning|ai|data science|analytics|tableau|power bi|excel|sap|salesforce|jira|confluence)\b/gi,
    /\b(leadership|communication|teamwork|problem.?solving|analytical|strategic|project management|stakeholder|collaboration|presentation|negotiation|decision.?making|time management|organizational|interpersonal|creative|innovative|detail.?oriented|self.?motivated|proactive)\b/gi,
    /\b(bachelor|master|mba|phd|certified|certification|pmp|cpa|cfa|aws certified|google certified|microsoft certified|scrum master|six sigma)\b/gi,
    /\b(\d+\+?\s*years?\s*(of\s*)?(experience|exp))\b/gi,
  ];

  const keywords = new Set();
  skillPatterns.forEach((pattern) => {
    const matches = jobDescription.match(pattern);
    if (matches) {
      matches.forEach((match) => keywords.add(match.toLowerCase().trim()));
    }
  });

  return Array.from(keywords);
}

// Calculate keyword match score
function calculateKeywordScore(text, keywords) {
  const textLower = text.toLowerCase();
  let matchedKeywords = [];
  let missingKeywords = [];

  keywords.forEach((keyword) => {
    if (textLower.includes(keyword.toLowerCase())) {
      matchedKeywords.push(keyword);
    } else {
      missingKeywords.push(keyword);
    }
  });

  const score =
    keywords.length > 0
      ? Math.round((matchedKeywords.length / keywords.length) * 100)
      : 0;

  return {
    score,
    matchedKeywords,
    missingKeywords,
    totalKeywords: keywords.length,
  };
}

// Analyze resume fit and generate score
async function analyzeResumeFit(
  resumeText,
  jobDescription,
  model = "gpt-4-turbo-preview",
) {
  try {
    const keywords = extractKeywords(jobDescription);
    const keywordAnalysis = calculateKeywordScore(resumeText, keywords);

    const prompt = `You are an expert HR professional, ATS (Applicant Tracking System) specialist, and resume analyst. Analyze the following resume against the job description comprehensively.

RESUME:
${resumeText}

JOB DESCRIPTION:
${jobDescription}

EXTRACTED KEYWORDS FROM JOB: ${keywords.join(", ")}
KEYWORD MATCH RATE: ${keywordAnalysis.score}%
MATCHED KEYWORDS: ${keywordAnalysis.matchedKeywords.join(", ")}
MISSING KEYWORDS: ${keywordAnalysis.missingKeywords.join(", ")}

Provide a detailed analysis including:
1. An overall fit score from 0-100 (consider both qualitative fit AND keyword matching - ATS systems heavily weight keyword matching)
2. A detailed explanation of the score
3. Key strengths that align with the job requirements
4. Critical gaps that need to be addressed
5. Specific keywords/skills from the job description that are missing or underrepresented
6. ATS compatibility assessment

Respond in JSON format:
{
  "score": <number>,
  "explanation": "<detailed string explaining the score>",
  "strengths": ["<specific strength 1>", "<specific strength 2>", ...],
  "gaps": ["<specific gap 1>", "<specific gap 2>", ...],
  "missingKeywords": ["<keyword1>", "<keyword2>", ...],
  "atsCompatibility": "<assessment of how well resume will perform in ATS>",
  "keywordMatchRate": <number>
}`;

    const completion = await openai.chat.completions.create({
      model: model,
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      response_format: { type: "json_object" },
    });

    const result = JSON.parse(completion.choices[0].message.content);
    result.extractedKeywords = keywords;
    result.keywordAnalysis = keywordAnalysis;

    return result;
  } catch (error) {
    console.error("Error analyzing resume fit:", error);
    if (
      error.message?.includes("content_policy") ||
      error.message?.includes("safety")
    ) {
      throw new Error(
        "Content was flagged by OpenAI's safety system. Please ensure your resume and job description contain professional content only.",
      );
    }
    throw error;
  }
}

// Generate comprehensive tailored resume
async function generateTailoredResume(
  resumeText,
  jobDescription,
  analysisResult,
  model = "gpt-4-turbo-preview",
) {
  try {
    const prompt = `You are an elite professional resume writer with 20+ years of experience crafting resumes that pass ATS systems and impress hiring managers. Your task is to create a COMPREHENSIVE, DETAILED, and HIGHLY TARGETED resume.

ORIGINAL RESUME:
${resumeText}

TARGET JOB DESCRIPTION:
${jobDescription}

INITIAL ANALYSIS:
- Current Score: ${analysisResult.score}/100
- Strengths: ${analysisResult.strengths.join(", ")}
- Gaps to Address: ${analysisResult.gaps.join(", ")}
- Missing Keywords to Include: ${analysisResult.missingKeywords?.join(", ") || "N/A"}
- All Job Keywords: ${analysisResult.extractedKeywords?.join(", ") || "N/A"}

CREATE A TRANSFORMED RESUME FOLLOWING THESE CRITICAL GUIDELINES:

1. **KEYWORD OPTIMIZATION**: 
   - Naturally integrate ALL missing keywords throughout the resume
   - Use exact phrases from the job description where appropriate
   - Include keywords in context (not just listed)
   - Ensure keywords appear in: Summary, Experience bullets, Skills section

2. **PROFESSIONAL SUMMARY** (4-5 powerful sentences):
   - Lead with years of experience and primary expertise area
   - Include 3-4 key skills mentioned in the job description
   - Mention a quantifiable achievement
   - Express alignment with the company's needs

3. **EXPERIENCE SECTION** - For EACH role, provide:
   - 5-7 detailed bullet points (not 2-3)
   - Start each bullet with a strong action verb
   - Include SPECIFIC metrics, percentages, dollar amounts, team sizes
   - **BOLD** the key skills/keywords within descriptions using **asterisks**
   - Show progression and increasing responsibility
   - Tie achievements directly to job requirements

4. **SKILLS SECTION**:
   - Organize into categories (Technical Skills, Soft Skills, Tools, Certifications)
   - Include ALL relevant keywords from the job description
   - List most relevant skills first

5. **ACHIEVEMENTS/PROJECTS** (if applicable):
   - Add a dedicated section for major accomplishments
   - Quantify impact wherever possible

6. **EDUCATION & CERTIFICATIONS**:
   - Include relevant coursework if applicable
   - List certifications that match job requirements

FORMAT REQUIREMENTS:
- Use **bold** for key terms, skills, and metrics that match the job description
- Make the resume 2-3 pages to be comprehensive (not a brief 1-pager)
- Use consistent formatting throughout
- Include clear section headers

CRITICAL INSTRUCTIONS:
- Output ONLY the resume content itself
- DO NOT include any explanatory text, disclaimers, or commentary
- DO NOT say things like "Here is the resume:" or "Due to complexity..." or "This is an outline..."
- DO NOT mention limitations or suggest further edits needed
- Start directly with the candidate's name or professional summary
- End with the last section (Education/Certifications)
- Provide a COMPLETE, FINISHED resume that is ready to use

The goal is to create a resume that will score 85+ when analyzed against this job description.`;

    const completion = await openai.chat.completions.create({
      model: model,
      messages: [
        {
          role: "system",
          content:
            "You are a professional resume writer. Output only the resume content with no explanations, disclaimers, or commentary. Never include phrases like 'Here is your resume' or 'Due to complexity'. Start directly with the resume.",
        },
        { role: "user", content: prompt },
      ],
      temperature: 0.7,
      max_tokens: 4000,
    });

    let resumeContent = completion.choices[0].message.content.trim();

    // Post-process to remove any remaining conversational text
    const conversationalPhrases = [
      /^(Here is|Here's|Below is|I've created|I've drafted|I will draft|Given the|Due to the complexity).*?[\n\r]+/i,
      /^.*?(resume|outline|template).*?follows?:?\s*[\n\r]+/i,
      /^.*?(However|Note that|Please note|Keep in mind).*?[\n\r]+/i,
      /For a complete.*?[\n\r]+/gi,
      /it's essential to.*?[\n\r]+/gi,
      /you can then expand.*?[\n\r]+/gi,
      /This (resume|outline|template).*?[\n\r]+/gi,
    ];

    conversationalPhrases.forEach((pattern) => {
      resumeContent = resumeContent.replace(pattern, "");
    });

    return resumeContent.trim();
  } catch (error) {
    console.error("Error generating tailored resume:", error);
    if (
      error.message?.includes("content_policy") ||
      error.message?.includes("safety")
    ) {
      throw new Error(
        "Content was flagged by OpenAI's safety system. Please ensure your resume and job description contain professional content only.",
      );
    }
    throw error;
  }
}

// Score the generated resume against the job description
async function scoreGeneratedResume(
  generatedResume,
  jobDescription,
  keywords,
  model = "gpt-4-turbo-preview",
) {
  try {
    const keywordAnalysis = calculateKeywordScore(generatedResume, keywords);

    const prompt = `You are an expert HR professional and ATS specialist. Score this GENERATED resume against the job description.

GENERATED RESUME:
${generatedResume}

JOB DESCRIPTION:
${jobDescription}

KEYWORD MATCH ANALYSIS:
- Match Rate: ${keywordAnalysis.score}%
- Matched Keywords: ${keywordAnalysis.matchedKeywords.join(", ")}
- Still Missing: ${keywordAnalysis.missingKeywords.join(", ")}

Evaluate how well this tailored resume now matches the job requirements. Provide:
1. A new fit score from 0-100
2. Assessment of improvements made
3. Any remaining gaps
4. ATS pass likelihood (percentage estimate)

Respond in JSON format:
{
  "score": <number>,
  "improvements": ["<improvement 1>", "<improvement 2>", ...],
  "remainingGaps": ["<gap 1>", "<gap 2>", ...],
  "atsPassLikelihood": <percentage>,
  "summary": "<brief summary of the transformation>",
  "keywordMatchRate": <number>
}`;

    const completion = await openai.chat.completions.create({
      model: model,
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      response_format: { type: "json_object" },
    });

    const result = JSON.parse(completion.choices[0].message.content);
    result.keywordAnalysis = keywordAnalysis;

    return result;
  } catch (error) {
    console.error("Error scoring generated resume:", error);

    // Check if it's a content policy/safety error
    if (
      error.message?.includes("content_policy") ||
      error.message?.includes("safety")
    ) {
      throw new Error(
        "The content was flagged by our safety system. Please ensure your resume and job description contain only professional information.",
      );
    }

    throw error;
  }
}

// Single clean professional template
function generateProfessionalHTML(resumeContent) {
  // Simple parse - just format the markdown content nicely
  const lines = resumeContent.split("\n");
  let html = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    @page { margin: 0.5in; }
    body {
      font-family: 'Segoe UI', -apple-system, BlinkMacSystemFont, 'Arial', sans-serif;
      font-size: 11pt;
      line-height: 1.65;
      color: #1f2937;
      background: white;
      padding: 40px 60px;
      max-width: 8.5in;
      margin: 0 auto;
    }
    h1 {
      font-size: 32pt;
      font-weight: 700;
      color: #1e3a5f;
      margin-bottom: 12px;
      text-transform: uppercase;
      letter-spacing: 3px;
      border-bottom: 3px solid #1e3a5f;
      padding-bottom: 12px;
    }
    h2 {
      font-size: 14pt;
      font-weight: 700;
      color: #1e3a5f;
      margin-top: 28px;
      margin-bottom: 14px;
      text-transform: uppercase;
      border-bottom: 2px solid #4a6fa5;
      padding-bottom: 6px;
      letter-spacing: 2px;
    }
    h3 {
      font-size: 12pt;
      font-weight: 600;
      color: #1e3a5f;
      margin-top: 16px;
      margin-bottom: 6px;
    }
    p {
      margin-bottom: 10px;
      color: #374151;
      line-height: 1.7;
    }
    .contact {
      color: #4a6fa5;
      font-size: 10.5pt;
      margin-bottom: 24px;
      line-height: 1.8;
    }
    .summary {
      background: #f8fafc;
      padding: 16px 20px;
      border-left: 4px solid #1e3a5f;
      margin-bottom: 20px;
      border-radius: 4px;
    }
    ul {
      margin-left: 24px;
      margin-bottom: 16px;
      margin-top: 8px;
    }
    li {
      margin-bottom: 8px;
      color: #374151;
      line-height: 1.6;
      padding-left: 4px;
    }
    li::marker {
      color: #1e3a5f;
      font-weight: bold;
    }
    strong, b {
      color: #1e3a5f;
      font-weight: 600;
    }
    .job-header {
      display: flex;
      justify-content: space-between;
      align-items: baseline;
      margin-top: 18px;
      margin-bottom: 8px;
    }
    .job-title {
      font-weight: 600;
      color: #1e3a5f;
      font-size: 12pt;
    }
    .job-company {
      font-weight: 500;
      color: #374151;
      font-size: 11pt;
      margin-bottom: 4px;
    }
    .dates {
      color: #6b7280;
      font-style: italic;
      font-size: 10pt;
      white-space: nowrap;
    }
    em, i {
      color: #6b7280;
      font-style: italic;
    }
  </style>
</head>
<body>
`;

  let inList = false;
  let isSummarySection = false;

  for (let line of lines) {
    line = line.trim();
    if (!line) continue;

    // Convert markdown to HTML
    if (line.startsWith("# ")) {
      if (inList) {
        html += "</ul>\n";
        inList = false;
      }
      html += `<h1>${line.substring(2).trim()}</h1>\n`;
    } else if (line.startsWith("## ")) {
      if (inList) {
        html += "</ul>\n";
        inList = false;
      }
      const sectionTitle = line.substring(3).trim();
      isSummarySection =
        sectionTitle.toLowerCase().includes("summary") ||
        sectionTitle.toLowerCase().includes("profile") ||
        sectionTitle.toLowerCase().includes("objective");
      html += `<h2>${sectionTitle}</h2>\n`;
    } else if (line.startsWith("### ")) {
      if (inList) {
        html += "</ul>\n";
        inList = false;
      }
      html += `<h3>${line.substring(4).trim()}</h3>\n`;
    } else if (
      line.startsWith("- ") ||
      line.startsWith("* ") ||
      line.startsWith("• ")
    ) {
      if (!inList) {
        html += "<ul>\n";
        inList = true;
      }
      const content = line
        .substring(2)
        .trim()
        .replace(/\*\*([^*]+)\*\*/g, "<strong>$1</strong>");
      html += `<li>${content}</li>\n`;
    } else if (line.match(/^[\w\s]+@[\w\.]+/)) {
      // Contact line
      html += `<p class="contact">${line}</p>\n`;
    } else if (line.startsWith("*") && line.endsWith("*") && !line.includes("**")) {
      // Italic line (job company/dates)
      if (inList) {
        html += "</ul>\n";
        inList = false;
      }
      const content = line.replace(/^\*+|\*+$/g, "").trim();
      html += `<p class="job-company"><em>${content}</em></p>\n`;
    } else {
      if (inList) {
        html += "</ul>\n";
        inList = false;
      }
      const content = line.replace(/\*\*([^*]+)\*\*/g, "<strong>$1</strong>");
      
      // Check if this looks like a job title with dates
      if (content.includes(" - ") && content.match(/\d{4}/)) {
        const parts = content.split(" - ");
        if (parts.length >= 2) {
          const title = parts.slice(0, -1).join(" - ");
          const dates = parts[parts.length - 1];
          html += `<div class="job-header">
            <span class="job-title">${title}</span>
            <span class="dates">${dates}</span>
          </div>\n`;
        } else {
          html += isSummarySection
            ? `<div class="summary"><p>${content}</p></div>\n`
            : `<p>${content}</p>\n`;
        }
      } else {
        html += isSummarySection
          ? `<div class="summary"><p>${content}</p></div>\n`
          : `<p>${content}</p>\n`;
      }
    }
  }

  if (inList) html += "</ul>\n";
  html += "</body>\n</html>";

  return html;
}

// Generate professional PDF using Puppeteer
async function generateProfessionalPDF(resumeContent, outputPath) {
  const html = generateProfessionalHTML(resumeContent);

  const browser = await puppeteer.launch({
    headless: true,
    args: ["--no-sandbox", "--disable-setuid-sandbox"],
  });

  try {
    const page = await browser.newPage();
    await page.setContent(html, { waitUntil: "networkidle0" });

    await page.pdf({
      path: outputPath,
      format: "Letter",
      printBackground: true,
      margin: { top: "0.5in", bottom: "0.5in", left: "0in", right: "0in" },
    });

    return outputPath;
  } finally {
    await browser.close();
  }
}

// Generate Two-Column Layout HTML (Template 1 - Marselina style)
function generateTwoColumnHTML(sections, style) {
  const name = sections.name || "Your Name";
  const title = sections.professionalTitle || "";
  const contact = sections.contact || [];

  // Clean text helper
  const clean = (text) => {
    if (!text) return "";
    return text
      .replace(/\*\*([^*]+)\*\*/g, "$1")
      .replace(/[%\u00c6]/g, "")
      .replace(/^[\u2022\-\*]\s*/, "")
      .trim();
  };

  // Build experience HTML
  let experienceHTML = "";
  if (sections.experience && sections.experience.length > 0) {
    sections.experience.forEach((job) => {
      const jobTitle = clean(job.title);
      // Parse company and dates from title
      const parts = jobTitle.split(/\s*[\|,]\s*/);
      const company = parts[0] || "";
      const position = parts[1] || "";
      const dates = parts[2] || "";

      experienceHTML += `
        <div class="experience-item">
          <div class="exp-left">
            <div class="company">${company}</div>
            <div class="dates">${dates}</div>
          </div>
          <div class="exp-right">
            <div class="position">${position || company}</div>
            <div class="description">
              ${
                job.bullets
                  ? job.bullets
                      .map((b) => clean(b))
                      .filter((b) => b)
                      .join(" ")
                  : ""
              }
            </div>
          </div>
        </div>
      `;
    });
  }

  // Build education HTML
  let educationHTML = "";
  if (sections.education && sections.education.length > 0) {
    sections.education.forEach((edu) => {
      const text = typeof edu === "string" ? clean(edu) : "";
      const parts = text.split(/\s*[\|,]\s*/);
      educationHTML += `
        <div class="education-item">
          <div class="edu-left">
            <div class="school">${parts[0] || ""}</div>
            <div class="dates">${parts[2] || ""}</div>
          </div>
          <div class="edu-right">
            <div class="degree">${parts[1] || text}</div>
          </div>
        </div>
      `;
    });
  }

  // Build skills HTML (two columns)
  let skillsHTML = "";
  if (sections.skills && sections.skills.length > 0) {
    const skills = sections.skills
      .map((s) => clean(typeof s === "string" ? s : ""))
      .filter((s) => s && !s.endsWith(":") && s.length > 1);

    const half = Math.ceil(skills.length / 2);
    const col1 = skills.slice(0, half);
    const col2 = skills.slice(half);

    skillsHTML = `
      <div class="skills-columns">
        <div class="skill-col">
          <div class="skill-header">Personal</div>
          ${col1.map((s) => `<div class="skill-item">${s}</div>`).join("")}
        </div>
        <div class="skill-col">
          <div class="skill-header">Professional</div>
          ${col2.map((s) => `<div class="skill-item">${s}</div>`).join("")}
        </div>
      </div>
    `;
  }

  // Contact info formatting
  const contactHTML = contact
    .map((c) => {
      const cleaned = clean(c);
      if (cleaned.includes("@"))
        return `<div class="contact-item">âœ‰ ${cleaned}</div>`;
      if (cleaned.match(/\d{3}.*\d{4}/))
        return `<div class="contact-item">ðŸ“ž ${cleaned}</div>`;
      if (cleaned.includes("linkedin") || cleaned.includes("www."))
        return `<div class="contact-item">ðŸŒ ${cleaned}</div>`;
      return `<div class="contact-item">ðŸ“ ${cleaned}</div>`;
    })
    .join("");

  return `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body {
      font-family: 'Segoe UI', Arial, sans-serif;
      font-size: 10pt;
      line-height: 1.4;
      color: #333;
      background: white;
    }
    .page {
      width: 8.5in;
      min-height: 11in;
      padding: 0.5in;
      background: white;
    }
    
    /* Header */
    .header {
      display: flex;
      justify-content: space-between;
      align-items: flex-start;
      margin-bottom: 30px;
      padding-bottom: 20px;
    }
    .header-left {
      flex: 1;
    }
    .name {
      font-size: 36pt;
      font-weight: 300;
      color: #1a1a1a;
      letter-spacing: 2px;
      line-height: 1.1;
      margin-bottom: 8px;
    }
    .title {
      font-size: 12pt;
      color: #666;
      font-weight: 400;
    }
    .header-right {
      text-align: right;
      font-size: 9pt;
      color: #555;
    }
    .contact-item {
      margin-bottom: 4px;
    }
    
    /* Sections */
    .section {
      margin-bottom: 25px;
    }
    .section-title {
      font-size: 11pt;
      font-weight: 600;
      color: #1a1a1a;
      letter-spacing: 3px;
      text-transform: uppercase;
      padding-bottom: 8px;
      margin-bottom: 15px;
      border-bottom: 2px solid #1a1a1a;
      display: inline-block;
      padding-right: 20px;
    }
    
    /* Experience */
    .experience-item, .education-item {
      display: flex;
      margin-bottom: 18px;
    }
    .exp-left, .edu-left {
      width: 150px;
      flex-shrink: 0;
      padding-right: 20px;
    }
    .exp-right, .edu-right {
      flex: 1;
    }
    .company, .school {
      font-weight: 500;
      color: #555;
      font-size: 9pt;
    }
    .dates {
      color: #888;
      font-size: 9pt;
    }
    .position, .degree {
      font-weight: 600;
      color: #1a1a1a;
      font-size: 11pt;
      margin-bottom: 5px;
    }
    .description {
      font-size: 9.5pt;
      color: #444;
      line-height: 1.5;
    }
    
    /* Skills */
    .skills-columns {
      display: flex;
      gap: 60px;
    }
    .skill-col {
      flex: 1;
    }
    .skill-header {
      font-weight: 600;
      font-size: 10pt;
      margin-bottom: 8px;
      color: #1a1a1a;
    }
    .skill-item {
      font-size: 9.5pt;
      color: #555;
      margin-bottom: 4px;
    }
    
    /* Summary */
    .summary-text {
      font-size: 10pt;
      color: #444;
      line-height: 1.6;
    }
  </style>
</head>
<body>
  <div class="page">
    <div class="header">
      <div class="header-left">
        <div class="name">${name.toUpperCase()}</div>
        <div class="title">${title}</div>
      </div>
      <div class="header-right">
        ${contactHTML}
      </div>
    </div>
    
    ${
      sections.summary && sections.summary.length > 0
        ? `
    <div class="section">
      <div class="section-title">About Me</div>
      <div class="summary-text">${sections.summary.map((s) => clean(s)).join(" ")}</div>
    </div>
    `
        : ""
    }
    
    ${
      experienceHTML
        ? `
    <div class="section">
      <div class="section-title">Work Experience</div>
      ${experienceHTML}
    </div>
    `
        : ""
    }
    
    ${
      educationHTML
        ? `
    <div class="section">
      <div class="section-title">Education</div>
      ${educationHTML}
    </div>
    `
        : ""
    }
    
    ${
      skillsHTML
        ? `
    <div class="section">
      <div class="section-title">Skill</div>
      ${skillsHTML}
    </div>
    `
        : ""
    }
  </div>
</body>
</html>
  `;
}

// Generate Centered Layout HTML (Template 2 - Sebastian style)
function generateCenteredHTML(sections, style) {
  const name = sections.name || "Your Name";
  const title = sections.professionalTitle || "";
  const contact = sections.contact || [];

  // Clean text helper
  const clean = (text) => {
    if (!text) return "";
    return text
      .replace(/\*\*([^*]+)\*\*/g, "$1")
      .replace(/[%\u00c6]/g, "")
      .replace(/^[\u2022\-\*]\s*/, "")
      .trim();
  };

  // Contact info inline
  const contactInline = contact
    .map((c) => {
      const cleaned = clean(c);
      if (!cleaned) return null;
      if (cleaned.includes("@"))
        return `<span class="contact-item">Email: ${cleaned}</span>`;
      if (cleaned.match(/\d{3}.*\d{4}/))
        return `<span class="contact-item">Phone: ${cleaned}</span>`;
      if (cleaned.includes("linkedin") || cleaned.includes("www."))
        return `<span class="contact-item">Link: ${cleaned}</span>`;
      return `<span class="contact-item">${cleaned}</span>`;
    })
    .filter(Boolean)
    .join('<span class="contact-sep">|</span>');

  // Build experience HTML
  let experienceHTML = "";
  if (sections.experience && sections.experience.length > 0) {
    sections.experience.forEach((job) => {
      const jobTitle = clean(job.title);
      const parts = jobTitle.split(/\s*[\|,]\s*/);

      experienceHTML += `
        <div class="experience-item">
          <div class="exp-header">
            <span class="company">${parts[0] || ""}</span>
            <span class="dates">${parts[2] || ""}</span>
          </div>
          <div class="position">${parts[1] || parts[0] || ""}</div>
          <div class="description">
            ${
              job.bullets
                ? job.bullets
                    .map((b) => clean(b))
                    .filter((b) => b)
                    .join(" ")
                : ""
            }
          </div>
        </div>
      `;
    });
  }

  // Build education HTML
  let educationHTML = "";
  if (sections.education && sections.education.length > 0) {
    sections.education.forEach((edu) => {
      const text = typeof edu === "string" ? clean(edu) : "";
      const parts = text.split(/\s*[\|,]\s*/);
      educationHTML += `
        <div class="education-item">
          <div class="edu-header">
            <span class="school">${parts[0] || ""}</span>
            <span class="dates">${parts[2] || ""}</span>
          </div>
          <div class="degree">${parts[1] || text}</div>
          <div class="edu-desc"></div>
        </div>
      `;
    });
  }

  // Build skills HTML (three columns with bullets)
  let skillsHTML = "";
  if (sections.skills && sections.skills.length > 0) {
    const skills = sections.skills
      .map((s) => clean(typeof s === "string" ? s : ""))
      .filter((s) => s && !s.endsWith(":") && s.length > 1);

    const third = Math.ceil(skills.length / 3);
    const cols = [
      skills.slice(0, third),
      skills.slice(third, third * 2),
      skills.slice(third * 2),
    ];

    skillsHTML = `
      <div class="skills-grid">
        ${cols
          .map(
            (col) => `
          <div class="skill-col">
            ${col.map((s) => `<div class="skill-item">â€¢ ${s}</div>`).join("")}
          </div>
        `,
          )
          .join("")}
      </div>
    `;
  }

  return `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body {
      font-family: 'Segoe UI', Arial, sans-serif;
      font-size: 11pt;
      line-height: 1.6;
      color: #333;
      background: white;
    }
    .page {
      width: 8.5in;
      min-height: 11in;
      background: white;
      position: relative;
    }
    
    /* Bars */
    .top-bar, .bottom-bar {
      height: 12px;
      background: #555;
      width: 100%;
    }
    .bottom-bar {
      position: absolute;
      bottom: 0;
    }
    
    .content {
      padding: 0.6in 0.75in;
    }
    
    /* Header */
    .header {
      text-align: center;
      margin-bottom: 30px;
      padding-bottom: 20px;
      border-bottom: 2px solid #333;
    }
    .name {
      font-size: 42pt;
      font-weight: 700;
      color: #1a1a1a;
      letter-spacing: 6px;
      margin-bottom: 8px;
    }
    .title {
      font-size: 14pt;
      color: #555;
      font-weight: 400;
      margin-bottom: 15px;
    }
    .contact-row {
      font-size: 10pt;
      color: #666;
    }
    .contact-item {
      margin: 0 12px;
    }
    .contact-sep {
      color: #ccc;
      margin: 0 8px;
    }
    
    /* Sections */
    .section {
      margin-bottom: 30px;
    }
    .section-title {
      font-size: 13pt;
      font-weight: 700;
      color: #1a1a1a;
      letter-spacing: 4px;
      text-transform: uppercase;
      padding-bottom: 10px;
      margin-bottom: 18px;
      border-bottom: 2px solid #555;
    }
    
    /* Experience */
    .experience-item, .education-item {
      margin-bottom: 20px;
    }
    .exp-header, .edu-header {
      display: flex;
      justify-content: space-between;
      margin-bottom: 5px;
    }
    .company, .school {
      font-size: 10pt;
      color: #888;
      font-style: italic;
    }
    .dates {
      font-size: 10pt;
      color: #888;
      font-style: italic;
    }
    .position, .degree {
      font-weight: 700;
      color: #1a1a1a;
      font-size: 12pt;
      margin-bottom: 8px;
    }
    .description, .edu-desc {
      font-size: 10.5pt;
      color: #444;
      line-height: 1.6;
    }
    
    /* Skills */
    .skills-grid {
      display: flex;
      gap: 40px;
    }
    .skill-col {
      flex: 1;
    }
    .skill-item {
      font-size: 10pt;
      color: #444;
      margin-bottom: 6px;
    }
    
    /* Summary */
    .summary-text {
      font-size: 11pt;
      color: #444;
      line-height: 1.7;
    }
  </style>
</head>
<body>
  <div class="page">
    <div class="top-bar"></div>
    <div class="content">
      <div class="header">
        <div class="name">${name.toUpperCase()}</div>
        <div class="title">${title}</div>
        <div class="contact-row">${contactInline}</div>
      </div>
      
      ${
        sections.summary && sections.summary.length > 0
          ? `
      <div class="section">
        <div class="section-title">About Me</div>
        <div class="summary-text">${sections.summary
          .map((s) => clean(s))
          .filter((s) => s)
          .join(" ")}</div>
      </div>
      `
          : ""
      }
      
      ${
        educationHTML
          ? `
      <div class="section">
        <div class="section-title">Education</div>
        ${educationHTML}
      </div>
      `
          : ""
      }
      
      ${
        experienceHTML
          ? `
      <div class="section">
        <div class="section-title">Work Experience</div>
        ${experienceHTML}
      </div>
      `
          : ""
      }
      
      ${
        skillsHTML
          ? `
      <div class="section">
        <div class="section-title">Skills</div>
        ${skillsHTML}
      </div>
      `
          : ""
      }
    </div>
    <div class="bottom-bar"></div>
  </div>
</body>
</html>
  `;
}

// Parse resume content into structured sections
function parseResumeContent(content) {
  const lines = content.split("\n");
  const sections = {
    name: "",
    professionalTitle: "",
    contact: [],
    summary: [],
    experience: [],
    skills: [],
    education: [],
    certifications: [],
    other: [],
  };

  let currentSection = "header";
  let currentJob = null;
  let headerLineCount = 0;

  const sectionPatterns = {
    summary: /^(professional\s+)?summary|profile|objective|about(\s+me)?/i,
    experience: /^(professional\s+)?experience|work\s+history|employment/i,
    skills: /^skills|technical\s+skills|core\s+competencies|expertise/i,
    education: /^education|academic/i,
    certifications: /^certifications?|licenses?|credentials/i,
    projects: /^projects|portfolio/i,
    awards: /^awards|achievements|honors/i,
  };

  // Common job title patterns
  const jobTitlePattern =
    /^(senior|lead|principal|staff|junior|associate|assistant|head|chief|director|manager|engineer|developer|designer|analyst|consultant|specialist|coordinator|administrator|executive|officer|vp|cto|ceo|cfo|coo)/i;

  lines.forEach((line, index) => {
    const trimmed = line.trim();
    if (!trimmed) return;

    // Check if this is a section header
    const cleanLine = trimmed.replace(/^#+\s*/, "").replace(/:$/, "");
    let foundSection = null;

    for (const [section, pattern] of Object.entries(sectionPatterns)) {
      if (pattern.test(cleanLine)) {
        foundSection = section;
        break;
      }
    }

    if (foundSection) {
      currentSection = foundSection;
      currentJob = null;
      return;
    }

    // Detect name (first substantial line that's not contact info)
    if (
      !sections.name &&
      index < 5 &&
      trimmed.length < 60 &&
      !trimmed.startsWith("â€¢") &&
      !trimmed.startsWith("-") &&
      !trimmed.match(/@|linkedin|github|phone|\d{3}[-.]?\d{3}/i)
    ) {
      sections.name = trimmed;
      currentSection = "contact";
      headerLineCount = 1;
      return;
    }

    // Detect professional title (second line after name, if it looks like a job title)
    if (
      sections.name &&
      !sections.professionalTitle &&
      headerLineCount === 1 &&
      index < 7 &&
      trimmed.length < 80 &&
      !trimmed.match(
        /@|linkedin|github|phone|\d{3}[-.]?\d{3}[-.]?\d{4}|www\.|\.com/i,
      ) &&
      (jobTitlePattern.test(trimmed) || trimmed.split(" ").length <= 6)
    ) {
      // Could be a professional title if it's short and doesn't look like contact info
      if (!trimmed.match(/\|/) && !trimmed.match(/[,].*[,]/)) {
        sections.professionalTitle = trimmed;
        headerLineCount = 2;
        return;
      }
    }

    // Detect contact info
    if (currentSection === "contact" || currentSection === "header") {
      if (
        trimmed.match(
          /@|linkedin|github|phone|\d{3}[-.]?\d{3}[-.]?\d{4}|www\.|\.com/i,
        )
      ) {
        // Split contact info if it contains multiple items separated by | or ,
        const contactItems = trimmed
          .split(/\s*[\|]\s*/)
          .filter((item) => item.trim());
        if (contactItems.length > 1) {
          sections.contact.push(...contactItems);
        } else {
          sections.contact.push(trimmed);
        }
        headerLineCount++;
        return;
      }
      // Check if we've moved past contact section
      if (sections.contact.length > 0 && !trimmed.match(/@|\d{3}/)) {
        currentSection = "summary";
      }
    }

    // Handle content based on current section
    switch (currentSection) {
      case "summary":
        sections.summary.push(trimmed);
        break;
      case "experience":
        // Check if this is a job title/company line
        if (
          !trimmed.startsWith("â€¢") &&
          !trimmed.startsWith("-") &&
          !trimmed.startsWith("*")
        ) {
          if (
            currentJob &&
            currentJob.bullets.length === 0 &&
            !trimmed.match(/\d{4}/)
          ) {
            // This might be a date or location line for current job
            currentJob.subtitle = trimmed;
          } else {
            currentJob = { title: trimmed, subtitle: "", bullets: [] };
            sections.experience.push(currentJob);
          }
        } else if (currentJob) {
          currentJob.bullets.push(trimmed.replace(/^[â€¢\-\*]\s*/, ""));
        } else {
          sections.experience.push({
            title: "",
            subtitle: "",
            bullets: [trimmed.replace(/^[â€¢\-\*]\s*/, "")],
          });
        }
        break;
      case "skills":
        sections.skills.push(trimmed);
        break;
      case "education":
        sections.education.push(trimmed);
        break;
      case "certifications":
        sections.certifications.push(trimmed);
        break;
      default:
        if (trimmed) {
          sections.other.push({ content: trimmed });
        }
    }
  });

  return sections;
}
// Generate styled DOCX - Premium Professional Layout
async function generateStyledDOCX(
  resumeContent,
  outputPath,
  styleName = "professional",
) {
  const style = RESUME_STYLES[styleName] || RESUME_STYLES.classic;

  // Parse resume into structured sections
  const sections = parseResumeContent(resumeContent);

  // Style configurations for DOCX - matching new templates
  const styleConfig = {
    classic: {
      headingColor: "2D2D2D",
      accentColor: "2D2D2D",
      bulletChar: "â€¢",
      textColor: "333333",
      layout: "centered",
    },
    modern: {
      headingColor: "1A1A1A",
      accentColor: "1A1A1A",
      bulletChar: "â€¢",
      textColor: "333333",
      layout: "two-column",
    },
    professional: {
      headingColor: "1E3A5F",
      accentColor: "1E3A5F",
      bulletChar: "â—",
      textColor: "2D3748",
      layout: "centered",
    },
    executive: {
      headingColor: "1F2937",
      accentColor: "1F2937",
      bulletChar: "â—†",
      textColor: "1F2937",
      layout: "centered",
    },
    minimal: {
      headingColor: "18181B",
      accentColor: "71717A",
      bulletChar: "â€”",
      textColor: "27272A",
      layout: "two-column",
    },
  };

  const config = styleConfig[styleName] || styleConfig.classic;
  const children = [];

  // Add name header (ALL CAPS with letter spacing for centered layout)
  if (sections.name) {
    children.push(
      new Paragraph({
        children: [
          new TextRun({
            text:
              config.layout === "centered"
                ? sections.name.toUpperCase()
                : sections.name,
            bold: true,
            size: 48,
            color: config.headingColor,
          }),
        ],
        alignment:
          config.layout === "centered"
            ? AlignmentType.CENTER
            : AlignmentType.LEFT,
        spacing: { after: 80 },
      }),
    );
  }

  // Add professional title if available
  if (sections.professionalTitle) {
    children.push(
      new Paragraph({
        children: [
          new TextRun({
            text: sections.professionalTitle,
            size: 22,
            color: "666666",
          }),
        ],
        alignment:
          config.layout === "centered"
            ? AlignmentType.CENTER
            : AlignmentType.LEFT,
        spacing: { after: 120 },
      }),
    );
  }

  // Add contact info
  if (sections.contact.length > 0) {
    children.push(
      new Paragraph({
        children: [
          new TextRun({
            text: sections.contact.join("     |     "),
            size: 20,
            color: "666666",
          }),
        ],
        alignment:
          config.layout === "centered"
            ? AlignmentType.CENTER
            : AlignmentType.LEFT,
        spacing: { after: 200 },
      }),
    );

    // Add divider line for centered layout styles
    if (config.layout === "centered") {
      children.push(
        new Paragraph({
          border: {
            bottom: {
              color: config.accentColor,
              space: 1,
              value: BorderStyle.SINGLE,
              size: 12,
            },
          },
          spacing: { after: 300 },
        }),
      );
    }
  }

  // Helper function to add a section
  const addSection = (title, content, isBulletList = false) => {
    if (!content || (Array.isArray(content) && content.length === 0)) return;

    // Section title with underline
    children.push(
      new Paragraph({
        children: [
          new TextRun({
            text: title,
            bold: true,
            size: 24,
            color: config.headingColor,
            allCaps: true,
          }),
        ],
        spacing: { before: 300, after: 100 },
        border: {
          bottom: {
            color: "CCCCCC",
            space: 1,
            value: BorderStyle.SINGLE,
            size: 4,
          },
        },
      }),
    );

    // Content
    if (Array.isArray(content)) {
      content.forEach((item) => {
        if (typeof item === "object" && item.title) {
          // Job entry
          children.push(
            new Paragraph({
              children: parseFormattedTextDOCX(item.title, config, true),
              spacing: { before: 150, after: 50 },
            }),
          );

          if (item.subtitle) {
            children.push(
              new Paragraph({
                children: [
                  new TextRun({
                    text: item.subtitle,
                    size: 20,
                    color: "666666",
                    italics: true,
                  }),
                ],
                spacing: { after: 80 },
              }),
            );
          }

          if (item.bullets && item.bullets.length > 0) {
            item.bullets.forEach((bullet) => {
              children.push(
                new Paragraph({
                  children: [
                    new TextRun({
                      text: `${config.bulletChar}  `,
                      color: config.headingColor,
                      size: 20,
                    }),
                    ...parseFormattedTextDOCX(bullet, config, false),
                  ],
                  spacing: { before: 60, after: 60 },
                  indent: { left: 400 },
                }),
              );
            });
          }
        } else if (typeof item === "string") {
          const trimmed = item.trim();
          if (
            trimmed.startsWith("â€¢") ||
            trimmed.startsWith("-") ||
            trimmed.startsWith("*")
          ) {
            children.push(
              new Paragraph({
                children: [
                  new TextRun({
                    text: `${config.bulletChar}  `,
                    color: config.headingColor,
                    size: 20,
                  }),
                  ...parseFormattedTextDOCX(
                    trimmed.replace(/^[â€¢\-\*]\s*/, ""),
                    config,
                    false,
                  ),
                ],
                spacing: { before: 60, after: 60 },
                indent: { left: 400 },
              }),
            );
          } else {
            children.push(
              new Paragraph({
                children: parseFormattedTextDOCX(trimmed, config, false),
                spacing: { before: 60, after: 60 },
              }),
            );
          }
        }
      });
    }
  };

  // Add all sections - using template-style titles
  if (sections.summary.length > 0) addSection("About Me", sections.summary);
  if (sections.experience.length > 0)
    addSection("Work Experience", sections.experience);
  if (sections.skills.length > 0) addSection("Skill", sections.skills);
  if (sections.education.length > 0)
    addSection("Education", sections.education);
  if (sections.certifications.length > 0)
    addSection("Certifications", sections.certifications);

  const doc = new Document({
    sections: [
      {
        properties: {
          page: {
            margin: {
              top: 720,
              right: 720,
              bottom: 720,
              left: 720,
            },
          },
        },
        children: children,
      },
    ],
  });

  const buffer = await Packer.toBuffer(doc);
  await fs.writeFile(outputPath, buffer);
}

// Helper function to parse text with ** bold ** formatting for DOCX
function parseFormattedTextDOCX(text, config, isTitle = false) {
  const parts = text.split(/(\*\*[^*]+\*\*)/g);
  const runs = [];

  parts.forEach((part) => {
    if (!part) return;

    if (part.startsWith("**") && part.endsWith("**")) {
      const boldText = part.slice(2, -2);
      runs.push(
        new TextRun({
          text: boldText,
          bold: true,
          color: config.headingColor,
          size: isTitle ? 24 : 21,
        }),
      );
    } else {
      runs.push(
        new TextRun({
          text: part,
          size: isTitle ? 24 : 21,
          color: config.textColor,
          bold: isTitle,
        }),
      );
    }
  });

  return runs;
}

// Generate styled Markdown - Beautiful formatted resume
function generateStyledMarkdown(resumeContent, styleName = "classic") {
  const style = RESUME_STYLES[styleName] || RESUME_STYLES.classic;
  const sections = parseResumeContent(resumeContent);

  let markdown = "";

  if (style.layout === "two-column") {
    // === Template 1 Style: Two-column layout ===

    // Name (large, bold)
    markdown += `# ${sections.name.toUpperCase()}\n`;

    // Professional title
    if (sections.professionalTitle) {
      markdown += `*${sections.professionalTitle}*\n\n`;
    }

    // Contact info
    if (sections.contact.length > 0) {
      markdown +=
        sections.contact.map((c) => c.replace(/[%\u00c6]/g, "")).join(" | ") +
        "\n\n";
    }

    markdown += `---\n\n`;

    // Sections
    const sectionOrder = [
      { key: "summary", title: "ABOUT ME" },
      { key: "experience", title: "WORK EXPERIENCE" },
      { key: "skills", title: "SKILL" },
      { key: "education", title: "EDUCATION" },
      { key: "certifications", title: "CERTIFICATIONS" },
    ];

    sectionOrder.forEach(({ key, title }) => {
      const content = sections[key];
      if (!content || (Array.isArray(content) && content.length === 0)) return;

      markdown += `## ${title}\n\n`;

      if (key === "experience" && Array.isArray(content)) {
        // Two-column style: Company/Dates | Title/Description
        content.forEach((job) => {
          const cleanTitle = job.title
            .replace(/\*\*([^*]+)\*\*/g, "$1")
            .replace(/[%\u00c6]/g, "")
            .trim();
          const titleParts = cleanTitle.split(/\s*[\|\u2013-]\s*/);
          const company = titleParts.length >= 2 ? titleParts[0] : "";
          const jobTitle =
            titleParts.length >= 2 ? titleParts.slice(1).join(" ") : cleanTitle;
          const dates = job.subtitle.replace(/[%\u00c6]/g, "").trim();

          if (company) markdown += `**${company}** | `;
          if (dates) markdown += `*${dates}*\n\n`;
          markdown += `**${jobTitle}**\n\n`;

          if (job.bullets && job.bullets.length > 0) {
            job.bullets.forEach((bullet) => {
              const cleanBullet = bullet
                .replace(/\*\*([^*]+)\*\*/g, "$1")
                .replace(/[%\u00c6]/g, "")
                .replace(/^[\u2022\-\*]\s*/, "");
              if (cleanBullet.trim()) {
                markdown += `â€¢ ${cleanBullet}\n`;
              }
            });
            markdown += `\n`;
          }
        });
      } else if (key === "education" && Array.isArray(content)) {
        let currentEntry = {
          institution: "",
          dates: "",
          degree: "",
          description: [],
        };
        const entries = [];

        content.forEach((line) => {
          const text =
            typeof line === "string"
              ? line
                  .replace(/\*\*([^*]+)\*\*/g, "$1")
                  .replace(/[%\u00c6]/g, "")
                  .trim()
              : "";
          if (!text) return;

          const dateMatch = text.match(
            /(\d{4}\s*[-\u2013]\s*\d{4}|\d{4}\s*[-\u2013]\s*Present|\d{4})/i,
          );
          const isInstitution = text.match(
            /university|college|school|institute/i,
          );
          const isDegree = text.match(
            /bachelor|master|doctor|mba|phd|b\.[a-z]+|m\.[a-z]+|degree|diploma/i,
          );

          if (isInstitution || (dateMatch && !isDegree)) {
            if (currentEntry.institution || currentEntry.degree)
              entries.push({ ...currentEntry });
            currentEntry = {
              institution: text,
              dates: "",
              degree: "",
              description: [],
            };
            if (dateMatch) {
              currentEntry.dates = dateMatch[0];
              currentEntry.institution = text
                .replace(dateMatch[0], "")
                .replace(/[\|,]\s*$/, "")
                .trim();
            }
          } else if (isDegree) {
            currentEntry.degree = text;
          } else {
            currentEntry.description.push(text);
          }
        });
        if (currentEntry.institution || currentEntry.degree)
          entries.push(currentEntry);

        entries.forEach((entry) => {
          if (entry.institution) markdown += `**${entry.institution}**`;
          if (entry.dates) markdown += ` | *${entry.dates}*`;
          markdown += `\n\n`;
          if (entry.degree) markdown += `***${entry.degree}***\n\n`;
          if (entry.description.length > 0) {
            markdown += `${entry.description.join(" ")}\n\n`;
          }
        });
      } else if (key === "skills" && Array.isArray(content)) {
        // Parse skills
        const categories = {};
        let currentCategory = null;

        content.forEach((line) => {
          const text =
            typeof line === "string"
              ? line
                  .replace(/\*\*([^*]+)\*\*/g, "$1")
                  .replace(/[%\u00c6]/g, "")
                  .trim()
              : "";
          if (!text) return;

          if (
            text.match(
              /^(Personal|Professional|Technical|Soft\s*Skills?|Hard\s*Skills?|Core)/i,
            ) ||
            text.endsWith(":")
          ) {
            currentCategory = text.replace(/:$/, "").trim();
            categories[currentCategory] = [];
          } else if (currentCategory) {
            categories[currentCategory].push(
              text.replace(/^[\u2022\-\*]\s*/, ""),
            );
          }
        });

        if (Object.keys(categories).length > 0) {
          Object.entries(categories).forEach(([catName, skills]) => {
            markdown += `**${catName}**\n\n`;
            skills.forEach((skill) => (markdown += `${skill}  \n`));
            markdown += `\n`;
          });
        }
      } else if (Array.isArray(content)) {
        content.forEach((item) => {
          const text =
            typeof item === "string"
              ? item
                  .replace(/\*\*([^*]+)\*\*/g, "$1")
                  .replace(/[%\u00c6]/g, "")
                  .trim()
              : "";
          if (text) markdown += `${text}\n\n`;
        });
      }
    });
  } else {
    // === Template 2 Style: Centered layout ===

    // Centered header
    markdown += `<div align="center">\n\n`;
    markdown += `# ${sections.name.toUpperCase()}\n\n`;

    if (sections.professionalTitle) {
      markdown += `*${sections.professionalTitle}*\n\n`;
    }

    // Contact info
    if (sections.contact.length > 0) {
      markdown +=
        sections.contact
          .map((c) => c.replace(/[%\u00c6]/g, ""))
          .join(" **|** ") + "\n\n";
    }

    markdown += `</div>\n\n`;
    markdown += `---\n\n`;

    // Sections
    const sectionOrder = [
      { key: "summary", title: "ABOUT ME" },
      { key: "experience", title: "WORK EXPERIENCE" },
      { key: "skills", title: "SKILL" },
      { key: "education", title: "EDUCATION" },
      { key: "certifications", title: "CERTIFICATIONS" },
    ];

    sectionOrder.forEach(({ key, title }) => {
      const content = sections[key];
      if (!content || (Array.isArray(content) && content.length === 0)) return;

      markdown += `## ${title}\n\n`;

      if (key === "experience" && Array.isArray(content)) {
        content.forEach((job) => {
          const cleanTitle = job.title
            .replace(/\*\*([^*]+)\*\*/g, "$1")
            .replace(/[%\u00c6]/g, "")
            .trim();
          const titleParts = cleanTitle.split(/\s*[\|\u2013-]\s*/);
          const company = titleParts.length >= 2 ? titleParts[0] : "";
          const jobTitle =
            titleParts.length >= 2 ? titleParts.slice(1).join(" ") : cleanTitle;
          const dates = job.subtitle.replace(/[%\u00c6]/g, "").trim();

          markdown += `**${company || jobTitle}**`;
          if (dates) markdown += ` | *${dates}*`;
          markdown += `\n\n`;

          if (company && jobTitle !== cleanTitle) {
            markdown += `***${jobTitle}***\n\n`;
          }

          if (job.bullets && job.bullets.length > 0) {
            job.bullets.forEach((bullet) => {
              const cleanBullet = bullet
                .replace(/\*\*([^*]+)\*\*/g, "$1")
                .replace(/[%\u00c6]/g, "")
                .replace(/^[\u2022\-\*]\s*/, "");
              if (cleanBullet.trim()) {
                markdown += `â€¢ ${cleanBullet}\n`;
              }
            });
            markdown += `\n`;
          }
        });
      } else if (key === "education" && Array.isArray(content)) {
        let currentEntry = {
          institution: "",
          dates: "",
          degree: "",
          description: [],
        };
        const entries = [];

        content.forEach((line) => {
          const text =
            typeof line === "string"
              ? line
                  .replace(/\*\*([^*]+)\*\*/g, "$1")
                  .replace(/[%\u00c6]/g, "")
                  .trim()
              : "";
          if (!text) return;

          const dateMatch = text.match(
            /(\d{4}\s*[-\u2013]\s*\d{4}|\d{4}\s*[-\u2013]\s*Present|\d{4})/i,
          );
          const isInstitution = text.match(
            /university|college|school|institute/i,
          );
          const isDegree = text.match(
            /bachelor|master|doctor|mba|phd|b\.[a-z]+|m\.[a-z]+|degree|diploma/i,
          );

          if (isInstitution || (dateMatch && !isDegree)) {
            if (currentEntry.institution || currentEntry.degree)
              entries.push({ ...currentEntry });
            currentEntry = {
              institution: text,
              dates: "",
              degree: "",
              description: [],
            };
            if (dateMatch) {
              currentEntry.dates = dateMatch[0];
              currentEntry.institution = text
                .replace(dateMatch[0], "")
                .replace(/[\|,]\s*$/, "")
                .trim();
            }
          } else if (isDegree) {
            currentEntry.degree = text;
          } else {
            currentEntry.description.push(text);
          }
        });
        if (currentEntry.institution || currentEntry.degree)
          entries.push(currentEntry);

        entries.forEach((entry) => {
          markdown += `**${entry.institution || ""}**`;
          if (entry.dates) markdown += ` | *${entry.dates}*`;
          markdown += `\n\n`;
          if (entry.degree) markdown += `***${entry.degree}***\n\n`;
          if (entry.description.length > 0) {
            markdown += `${entry.description.join(" ")}\n\n`;
          }
        });
      } else if (key === "skills" && Array.isArray(content)) {
        // Flat bullet list
        content.forEach((line) => {
          const text =
            typeof line === "string"
              ? line
                  .replace(/\*\*([^*]+)\*\*/g, "$1")
                  .replace(/[%\u00c6]/g, "")
                  .trim()
              : "";
          if (!text) return;
          if (
            !text.endsWith(":") &&
            !text.match(/^(Personal|Professional|Technical)/i)
          ) {
            markdown += `â€¢ ${text.replace(/^[\u2022\-\*]\s*/, "")}\n`;
          }
        });
        markdown += `\n`;
      } else if (Array.isArray(content)) {
        content.forEach((item) => {
          const text =
            typeof item === "string"
              ? item
                  .replace(/\*\*([^*]+)\*\*/g, "$1")
                  .replace(/[%\u00c6]/g, "")
                  .trim()
              : "";
          if (text) markdown += `${text}\n\n`;
        });
      }
    });
  }

  return markdown;
}

// IPC Handlers
ipcMain.handle("select-file", async () => {
  const result = await dialog.showOpenDialog(mainWindow, {
    properties: ["openFile"],
    filters: [{ name: "Resume Files", extensions: ["pdf", "docx"] }],
  });

  if (!result.canceled && result.filePaths.length > 0) {
    const filePath = result.filePaths[0];
    const fileName = path.basename(filePath);
    return { filePath, fileName };
  }
  return null;
});

ipcMain.handle(
  "analyze-resume",
  async (
    event,
    { filePath, jobDescription, model = "gpt-4-turbo-preview" },
  ) => {
    try {
      // Extract text from resume
      const resumeText = await extractText(filePath);

      // Analyze initial resume fit
      const initialAnalysis = await analyzeResumeFit(
        resumeText,
        jobDescription,
        model,
      );

      // Generate tailored resume
      const tailoredResume = await generateTailoredResume(
        resumeText,
        jobDescription,
        initialAnalysis,
        model,
      );

      // Score the generated resume
      const finalScore = await scoreGeneratedResume(
        tailoredResume,
        jobDescription,
        initialAnalysis.extractedKeywords || [],
        model,
      );

      return {
        success: true,
        initialAnalysis,
        tailoredResume,
        finalScore,
      };
    } catch (error) {
      console.error("Error in analyze-resume:", error);
      throw error;
    }
  },
);

ipcMain.handle("save-pdf", async (event, { content, style }) => {
  try {
    const timestamp = new Date().toISOString().replace(/:/g, "-").split(".")[0];
    const defaultName = `Tailored_Resume_${timestamp}.pdf`;

    const result = await dialog.showSaveDialog(mainWindow, {
      defaultPath: defaultName,
      filters: [{ name: "PDF Files", extensions: ["pdf"] }],
    });

    if (!result.canceled && result.filePath) {
      // Check if file exists and is locked
      try {
        await generateProfessionalPDF(content, result.filePath);
        return { success: true, path: result.filePath };
      } catch (error) {
        if (error.code === "EBUSY" || error.code === "EPERM") {
          // File is locked (probably open in a PDF reader)
          return {
            success: false,
            error:
              "The file is currently open in another program. Please close it and try again, or save with a different name.",
            locked: true,
          };
        }
        throw error;
      }
    }
    return { success: false };
  } catch (error) {
    console.error("Error in save-pdf:", error);
    return {
      success: false,
      error: error.message || "Failed to save PDF",
    };
  }
});

ipcMain.handle("save-html", async (event, { content, style }) => {
  try {
    const timestamp = new Date().toISOString().replace(/:/g, "-").split(".")[0];
    const defaultName = `Tailored_Resume_${timestamp}.html`;

    const result = await dialog.showSaveDialog(mainWindow, {
      defaultPath: defaultName,
      filters: [{ name: "HTML Files", extensions: ["html"] }],
    });

    if (!result.canceled && result.filePath) {
      try {
        const html = generateProfessionalHTML(content);
        await fs.writeFile(result.filePath, html, "utf8");
        return { success: true, path: result.filePath };
      } catch (error) {
        if (error.code === "EBUSY" || error.code === "EPERM") {
          return {
            success: false,
            error:
              "The file is currently open in another program. Please close it and try again, or save with a different name.",
            locked: true,
          };
        }
        throw error;
      }
    }
    return { success: false };
  } catch (error) {
    console.error("Error in save-html:", error);
    return {
      success: false,
      error: error.message || "Failed to save HTML",
    };
  }
});

ipcMain.handle("save-docx", async (event, { content, style }) => {
  try {
    const timestamp = new Date().toISOString().replace(/:/g, "-").split(".")[0];
    const defaultName = `Tailored_Resume_${timestamp}.docx`;

    const result = await dialog.showSaveDialog(mainWindow, {
      defaultPath: defaultName,
      filters: [{ name: "Word Documents", extensions: ["docx"] }],
    });

    if (!result.canceled && result.filePath) {
      try {
        await generateStyledDOCX(
          content,
          result.filePath,
          style || "professional",
        );
        return { success: true, path: result.filePath };
      } catch (error) {
        if (error.code === "EBUSY" || error.code === "EPERM") {
          return {
            success: false,
            error:
              "The file is currently open in another program. Please close it and try again, or save with a different name.",
            locked: true,
          };
        }
        throw error;
      }
    }
    return { success: false };
  } catch (error) {
    console.error("Error in save-docx:", error);
    return {
      success: false,
      error: error.message || "Failed to save DOCX",
    };
  }
});

ipcMain.handle("save-markdown", async (event, { content, style }) => {
  try {
    const timestamp = new Date().toISOString().replace(/:/g, "-").split(".")[0];
    const defaultName = `Tailored_Resume_${timestamp}.md`;

    const result = await dialog.showSaveDialog(mainWindow, {
      defaultPath: defaultName,
      filters: [{ name: "Markdown", extensions: ["md"] }],
    });

    if (!result.canceled && result.filePath) {
      try {
        const markdown = generateStyledMarkdown(content, style || "classic");
        await fs.writeFile(result.filePath, markdown, "utf8");
        return { success: true, path: result.filePath };
      } catch (error) {
        if (error.code === "EBUSY" || error.code === "EPERM") {
          return {
            success: false,
            error:
              "The file is currently open in another program. Please close it and try again, or save with a different name.",
            locked: true,
          };
        }
        throw error;
      }
    }
    return { success: false };
  } catch (error) {
    console.error("Error in save-markdown:", error);
    return {
      success: false,
      error: error.message || "Failed to save Markdown",
    };
  }
});

ipcMain.handle("get-pdf-styles", async () => {
  return Object.entries(RESUME_STYLES).map(([key, value]) => ({
    id: key,
    name: value.name,
    colors: value.colors,
  }));
});

ipcMain.handle("generate-preview-html", async (event, { content }) => {
  try {
    const html = generateProfessionalHTML(content);
    return { success: true, html };
  } catch (error) {
    console.error("Error generating preview HTML:", error);
    return { success: false, error: error.message };
  }
});

ipcMain.handle("save-text", async (event, { content }) => {
  try {
    const timestamp = new Date().toISOString().replace(/:/g, "-").split(".")[0];
    const defaultName = `Tailored_Resume_${timestamp}.txt`;

    const result = await dialog.showSaveDialog(mainWindow, {
      defaultPath: defaultName,
      filters: [{ name: "Text Files", extensions: ["txt"] }],
    });

    if (!result.canceled && result.filePath) {
      try {
        await fs.writeFile(result.filePath, content, "utf8");
        return { success: true, path: result.filePath };
      } catch (error) {
        if (error.code === "EBUSY" || error.code === "EPERM") {
          return {
            success: false,
            error:
              "The file is currently open in another program. Please close it and try again, or save with a different name.",
            locked: true,
          };
        }
        throw error;
      }
    }
    return { success: false };
  } catch (error) {
    console.error("Error in save-text:", error);
    return {
      success: false,
      error: error.message || "Failed to save text file",
    };
  }
});
